<?php

namespace App\Http\Controllers;

use App\User;
use App\Customer;
use App\Mail\ForgetPasswordMail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;

class AuthController extends Controller
{
    public function register(Request $request)
    {
        $fields = Validator::make($request->all(), [
            'name' => 'required|string',
            'email' => 'required|string|unique:users,email',
            'password' => 'required|string',
            'phone' => 'required|string',
            'password_confirmation' => 'required|same:password'
        ]);

        if ($fields->fails()) {
            return response()->json(['error' => $fields->errors()], 400);
        }

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => bcrypt($request->password),
            'phone' => $request->phone,
            'role' => $request->role
        ]);

        $token = $user->createToken('myapptoken')->plainTextToken;

        $response = [
            'user' => $user,
            'token' => $token
        ];

        return response($response, 201);
    }

    public function registerCustomer(Request $request)
    {
        $fields = Validator::make($request->all(), [
            'name' => 'required|string',
            'email' => 'required|string|unique:users,email',
            'password' => 'required|string',
            'password_confirmation' => 'required|same:password',
        ]);

        if ($fields->fails()) {
            return response()->json(['error' => $fields->errors()], 400);
        }



        $user = Customer::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => bcrypt($request->password),
            'phone' => $request->phone,
            'role' => $request->role
        ]);

        $token = $user->createToken('myapptoken')->plainTextToken;

        $response = [
            'customer' => $user,
            'token' => $token
        ];

        return response($response, 201);
    }

     public function login(Request $request)
    {
        $fields = Validator::make($request->all(), [
            'email' => 'required|string',
            'password' => 'required|string',
        ]);
        

        if ($fields->fails()) {
            return response()->json(['error' => $fields->errors()], 400);
        }

        // Check email
        $user = User::where('email', $request->email)->first();
        // return response(["mypass"=>encrypt($request->password),"alrea"=>encrypt($user->password)]);

        // Check password
        if (!$user || !Hash::check($request->password, $user->password)) {
            return response([
                'message' => 'Bad creds'
            ], 401);
        }

        $token = $user->createToken('myapptoken')->plainTextToken;

        $response = [
            'user' => $user,
            'token' => $token
        ];

        return response($response, 201);
    }

    public function loginCustomer(Request $request)
    {
        $fields = Validator::make($request->all(), [
            'email' => 'required|string',
            'password' => 'required|string'
        ]);

        if ($fields->fails()) {
            return response()->json(['error' => $fields->errors()], 400);
        }

        // Check email
        $user = Customer::where('email', $request->email)->first();

        // Check password
        if (!$user || !Hash::check($request->password, $user->password)) {
            return response([
                'message' => 'Bad creds'
            ], 401);
        }

        $token = $user->createToken('myapptoken')->plainTextToken;

        $response = [
            'customer' => $user,
            'token' => $token
        ];

        return response($response, 201);
    }

    public function logout(Request $request)
    {
        auth()->user()->tokens()->delete();

        return [
            'message' => 'Logged out'
        ];
    }
  
   public function forgetPassword(Request $request)
    {
        $fields = Validator::make($request->all(), [
            'email' => 'required|string|email',
        ]);

        if ($fields->fails()) {

            return response()->json(['error' => $fields->errors()], 422);
        }
        $user = User::where('email', $request->email)->first();

        if (isset($user)) {
            $user->forget_password_token = Str::random(32);
            Mail::to($user->email)->send(new ForgetPasswordMail($user));
            $user->save();

            $response = [
                'code' => 200,
                'status' => 'success',
                'status_message' => 'Password reset link has been sent to your registered email id. Click on it to reset your password.'
            ];

            return response()->json($response);
        }

        $response = [
            'code' => 404,
            'status' => 'failed',
            'status_message' => 'User not found.'
        ];

        return response($response);
    }

    public function updatePassword(Request $request)
    {
        $fields = Validator::make($request->all(), [
            'token' => 'required|string',
            'password' => 'required|string',
            'confirmation_password' => 'required|string|same:password'
        ]);

        if ($fields->fails()) {
            return response()->json(['error' => $fields->errors()], 422);
        }

        $user = User::where('forget_password_token', $request->token)->first();

        // if ($user->is_password_changed == 1) {
        //     return response()->json($arrayName = array('error' => 'your password is already been changed.'));
        // }

        if ($user) {
            $user->password = Hash::make($request->password);
            $user->save();
            return response()->json($arrayName = array('success' => 'password changed'));
        }

        return response()->json($arrayName = array('error' => 'user does not exists or token expired'));
    }
}
